<?php

/**
 * Class utilizing some handy methods for string transformation.
 * @author Imaginalis Software TM
 * @link http://imaginalis.pl
 */
class StringUtils
{
	/**
	 * Shortens a string.
	 * @param string $string String to perform transformation.
	 * @param integer $maxLength Maximum length of the shortened string.
	 * @param boolean $preserveLastWord Determines whether string shall be cut preserving last word in it.
	 * @param string $suffix Suffix for the shortened string.
	 * @return string Shortened string.
	 */
	public static function shortenString($string, $maxLength, $preserveLastWord = true, $suffix = '')
	{
		if(mb_strlen($string, 'UTF-8') > $maxLength)
		{
			if($preserveLastWord)
				return preg_replace('/\s+?(\S+)?$/', '', mb_substr($string, 0, $maxLength - mb_strlen($suffix, 'UTF-8'), 'UTF-8')).$suffix;
			else
				return mb_substr($string, 0, $maxLength - mb_strlen($suffix, 'UTF-8'), 'UTF-8').$suffix;
		}

		return $string;
	}

	/**
	 * Transforms diacritics of a string to their ASCII equivalents.
	 * @param string $string String to perform transformation.
	 * @return string Diacritics-free string.
	 */
	public static function transformDiactritics($string)
	{
	   $from = array('Á', 'À', 'Â', 'Ä', 'Ă', 'Ā', 'Ã', 'Å', 'Ą', 'Æ', 'Ć', 'Ċ', 'Ĉ', 'Č', 'Ç', 'Ď', 'Đ', 'Ð', 'É', 'È', 'Ė', 'Ê', 'Ë', 'Ě', 'Ē', 'Ę', 'Ə', 'Ġ', 'Ĝ', 'Ğ', 'Ģ', 'á', 'à', 'â', 'ä', 'ă', 'ā', 'ã', 'å', 'ą', 'æ', 'ć', 'ċ', 'ĉ', 'č', 'ç', 'ď', 'đ', 'ð', 'é', 'è', 'ė', 'ê', 'ë', 'ě', 'ē', 'ę', 'ə', 'ġ', 'ĝ', 'ğ', 'ģ', 'Ĥ', 'Ħ', 'I', 'Í', 'Ì', 'İ', 'Î', 'Ï', 'Ī', 'Į', 'Ĳ', 'Ĵ', 'Ķ', 'Ļ', 'Ł', 'Ń', 'Ň', 'Ñ', 'Ņ', 'Ó', 'Ò', 'Ô', 'Ö', 'Õ', 'Ő', 'Ø', 'Ơ', 'Œ', 'ĥ', 'ħ', 'ı', 'í', 'ì', 'i', 'î', 'ï', 'ī', 'į', 'ĳ', 'ĵ', 'ķ', 'ļ', 'ł', 'ń', 'ň', 'ñ', 'ņ', 'ó', 'ò', 'ô', 'ö', 'õ', 'ő', 'ø', 'ơ', 'œ', 'Ŕ', 'Ř', 'Ś', 'Ŝ', 'Š', 'Ş', 'Ť', 'Ţ', 'Þ', 'Ú', 'Ù', 'Û', 'Ü', 'Ŭ', 'Ū', 'Ů', 'Ų', 'Ű', 'Ư', 'Ŵ', 'Ý', 'Ŷ', 'Ÿ', 'Ź', 'Ż', 'Ž', 'ŕ', 'ř', 'ś', 'ŝ', 'š', 'ş', 'ß', 'ť', 'ţ', 'þ', 'ú', 'ù', 'û', 'ü', 'ŭ', 'ū', 'ů', 'ų', 'ű', 'ư', 'ŵ', 'ý', 'ŷ', 'ÿ', 'ź', 'ż', 'ž');
	   $to = array('A', 'A', 'A', 'A', 'A', 'A', 'A', 'A', 'A', 'AE', 'C', 'C', 'C', 'C', 'C', 'D', 'D', 'D', 'E', 'E', 'E', 'E', 'E', 'E', 'E', 'E', 'G', 'G', 'G', 'G', 'G', 'a', 'a', 'a', 'a', 'a', 'a', 'a', 'a', 'a', 'ae', 'c', 'c', 'c', 'c', 'c', 'd', 'd', 'd', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'e', 'g', 'g', 'g', 'g', 'g', 'H', 'H', 'I', 'I', 'I', 'I', 'I', 'I', 'I', 'I', 'IJ', 'J', 'K', 'L', 'L', 'N', 'N', 'N', 'N', 'O', 'O', 'O', 'O', 'O', 'O', 'O', 'O', 'CE', 'h', 'h', 'i', 'i', 'i', 'i', 'i', 'i', 'i', 'i', 'ij', 'j', 'k', 'l', 'l', 'n', 'n', 'n', 'n', 'o', 'o', 'o', 'o', 'o', 'o', 'o', 'o', 'o', 'R', 'R', 'S', 'S', 'S', 'S', 'T', 'T', 'T', 'U', 'U', 'U', 'U', 'U', 'U', 'U', 'U', 'U', 'U', 'W', 'Y', 'Y', 'Y', 'Z', 'Z', 'Z', 'r', 'r', 's', 's', 's', 's', 'B', 't', 't', 'b', 'u', 'u', 'u', 'u', 'u', 'u', 'u', 'u', 'u', 'u', 'w', 'y', 'y', 'y', 'z', 'z', 'z');

	   $cyrylicFrom = array('А', 'Б', 'В', 'Г', 'Д', 'Е', 'Ё', 'Ж', 'З', 'И', 'Й', 'К', 'Л', 'М', 'Н', 'О', 'П', 'Р', 'С', 'Т', 'У', 'Ф', 'Х', 'Ц', 'Ч', 'Ш', 'Щ', 'Ъ', 'Ы', 'Ь', 'Э', 'Ю', 'Я', 'а', 'б', 'в', 'г', 'д', 'е', 'ё', 'ж', 'з', 'и', 'й', 'к', 'л', 'м', 'н', 'о', 'п', 'р', 'с', 'т', 'у', 'ф', 'х', 'ц', 'ч', 'ш', 'щ', 'ъ', 'ы', 'ь', 'э', 'ю', 'я');
	   $cyrylicTo = array('A', 'B', 'W', 'G', 'D', 'Ie', 'Io', 'Z', 'Z', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'R', 'S', 'T', 'U', 'F', 'Ch', 'C', 'Tch', 'Sh', 'Shtch', '', 'Y', '', 'E', 'Iu', 'Ia', 'a', 'b', 'w', 'g', 'd', 'ie', 'io', 'z', 'z', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'r', 's', 't', 'u', 'f', 'ch', 'c', 'tch', 'sh', 'shtch', '', 'y', '', 'e', 'iu', 'ia');

	   $from = array_merge($from, $cyrylicFrom);
	   $to = array_merge($to, $cyrylicTo);

	   return str_replace($from, $to, $string);
	}

	/**
	 * Transforms specific string to its SEO Friendly form.
	 * @param string $string String to perform transformation.
	 * @param string $separator String used for separating words.
	 * @return string SEO Friendly string.
	 */
	public static function seoFriendlyString($string, $separator = '-')
	{
		if(!preg_match('/^[a-zA-Z0-9]+[a-zA-Z0-9\_\-]*$/', $string)) //check whether string is SEO Friendly already
		{
			$string = self::transformDiactritics($string); //replaces all diacritics to their ASCII equivalents

			$string = preg_replace('/[^A-Za-z0-9-]/', ' ', $string); //change all characters that don't match to spaces
			$string = preg_replace('/ +/', ' ', $string); //remove redundant spaces
			$string = trim($string); //trim spaces from the beggining and the end of the string
			$string = str_replace(' ', '-', $string); //transform spaces to hyphens
			$string = preg_replace('/-+/', '-', $string); //remove redundant hyphens
		}

		if($separator != '-')
			$string = str_replace('-', $separator, $string);

		return strtolower($string);
	}

	/**
	 * Zwraca wyraz jednostki dla zadanej liczby. Przed wykonaniem konwertuje do int.
	 * @param int|float|string $number np. 23
	 * @param string[] $units np. array('metr', 'metry', 'metrów')
	 * @return string np. dla 23 zwraca metrów, dla 1 - metr dla 3 - metry.
	 */
	public static function numberUnitWorld($number, $units)
	{
		$number = abs((int)$number);
		if ($number == 1)
			return $units[0];
		else if (($number < 10 || $number > 20) && ($number%10) > 1 && ($number%10) < 5)
			return $units[1];
		else
			return $units[2];
	}

	/**
	 * Zwraca liczbe w postaci słownej. Przed wykonaniem konwertuje do integer.
	 * @param int|float|string $number
	 * @return string
	 */
	public static function numberToWorld($number)
	{
		if (!is_numeric($number))
			throw new Exception('Invalid argument');

		$number = (int)$number;

		static $jednosci = array( 'zero', 'jeden', 'dwa', 'trzy', 'cztery', 'pięć', 'sześć', 'siedem', 'osiem', 'dziewięć',
			'dziesięć', 'jedenaście', 'dwanaście', 'trzynaście', 'czternaście', 'piętnaście', 'szesnaście', 'siedemnaście', 'osiemnaście', 'dzięwiętnaście');
		static $dziesiatki = array( '', 'dziesięć', 'dwadzieścia', 'trzydzieści', 'czterdzieści', 'piećdziesiąt', 'sześćdziesiąt', 'siedemdziesiąt', 'osiemdziesiąt', 'dziewięćdziesiąt' );
		static $setki = array( '', 'sto', 'dwieście', 'trzysta', 'czterysta', 'pięćset', 'sześćset', 'siedemset', 'osiemset', 'dziewięćset' );
		static $units = array(
			array( '', '', ''),
			array( 'tysiąc', 'tysiące', 'tysięcy'),
			array( 'milion', 'miliony', 'milionów'),
			array( 'miliard', 'miliardy', 'miliardów'),
			array( 'bilion', 'biliony', 'bilionów'),
			array( 'biliard', 'biliardy', 'biliardów'),
			array( 'trylion', 'tryliony', 'trylionów'),
		);

		$sign = $number < 0;
		$number = (int)(abs($number));

		if ($number < 20)
			return $sign ? 'minus ' . $jednosci[$number] : $jednosci[$number];

		$result = '';

		$group = 0;

		while ($number > 0 && $group < count($units)) {

			$n1000 = (int)($number % 1000);

			//handle houndreds
			$a = (int)(($n1000 / 100) % 10);
			$tmp = $setki[$a] . ' ';

			//handle < 100
			$n100 = (int)($n1000 % 100);
			if ($n100 < 20) {
				if ($n100 > 1 || ($number > 1 && $n100 > 0))	// jeden tysiąć => tysiąć | sto zero => sto
					$tmp .= $jednosci[$n100];
			}
			else{
				$a = (int)($n100 % 10);
				$b = (int)($n100 / 10);
				$tmp .= $dziesiatki[$b];
				if ($a)	//osiemdziesiąt zero => osiemdziesiąt
					$tmp .= ' ' . $jednosci[$a];
			}

			if ($n1000)	//dwa tysiące zero => dwa tysiące
				$result = trim($tmp) . ' ' . self::numberUnitWorld($number, $units[$group]) . ' ' . $result;
			$number = (int)($number / 1000);
			$group++;
		}

		return $sign ? 'minus ' . trim($result) : trim($result);
	}

	/**
	 * Konwertuje ciągi zanków zlezne od pułci użytkownika
	 *
	 * eg. "Użytkownik jest [kobietą/mężczyzną]."
	 * eg. "Użytkownik jest [kobietą\mężczyzną]."
	 * eg. "Użytkownik jest [kobietą|mężczyzną]."
	 *
	 * @param type $message
	 * @return type
	 */
	public static function c($message, $sex = null)
	{
		if (empty($sex))
			$sex = Yii::app()->user->getState('plec');
		$replacement = $sex == 'k' ? '$1' : '$2';
		$result = preg_replace('/\\[([^\\[\\]\\\\\\/\\|]*)[\\\\\\/\\|]([^\\[\\]\\\\\\/\\|]*)\\]/i', $replacement, $message);
		return $result;
	}

	/**
	 * Zwraca interwał między czasem obecnym a podanym
	 * Gdy [podany] < [obecengo] => '[interwał] temu'
	 * Gdy [podany] > [obecnego] => 'za [interwał]'
	 * @param string|int|DateTimeInterface $datetime
	 * @param string|int|DateTimeInterface $refrenceDate @default time()
	 * @return string
	 */
	public static function timeIntervalString($datetime, $refrenceDate = null)
	{
		$time_ts = is_string($datetime) ? strtotime($datetime)
			: (is_int($datetime) ? $datetime
			: ($datetime instanceof DateTimeInterface ? $datetime->getTimestamp()
				: null));

		if (!isset($time_ts))
			throw new Exception('Invalid argument');

		$refrence_ts = is_string($refrenceDate) ? strtotime($refrenceDate)
			: (is_int($refrenceDate) ? $refrenceDate
			: ($refrenceDate instanceof DateTimeInterface ? $refrenceDate->getTimestamp()
				: time()));

		static $data = array(
			'minutę'      => 120, // 2 * 60
			'%i minuty'   => 300, // 5 * 60
			'%i minut'    => 3600, // 60 * 60
			'godzinę'     => 7200, // 2 * 60 * 60
			'%h godziny'  => 18000, // 5 * 60 * 60
			'%h godzin'   => 86400, // 24 * 60 * 60
			'dzień'       => 172800, // 2 * 24 * 60 * 60
			'%d dni'      => 518400, // 5 * 24 * 60 * 60
			'tydzeń'      => 691200, // 8 * 24 * 60 * 60
			'%d dni'      => 2592000, // 30 * 24 * 60 * 60
			'miesiąc'     => 5184000, // 2 * 30 * 24 * 60 * 60
			'%m miesiące' => 12960000, // 5 * 30 * 24 * 60 * 60
			'%m miesięcy' => 31449600, // 364 * 24 * 60 * 60
			'rok'         => 62899200, // 2 * 364 * 24 * 60 * 60
			'%y lata'     => 157248000, // 5 * 364 * 24 * 60 * 60
		);

		$interval_ts = abs($refrence_ts - $time_ts);

		$pattern = '%y lat';
		foreach($data as $timePattern => $interval_sec) {
			if ($interval_ts  < $interval_sec) {
				$pattern = $timePattern;
				break;
			}
		}

		$refrenceDate = new DateTime();
		$refrenceDate->setTimestamp($refrence_ts);

		$datetime = new DateTime();
		$datetime->setTimestamp($time_ts);

		$interval = $datetime->diff($refrenceDate);

		$str = $interval->format($pattern);

		return $interval->invert ? ('za ' . $str) : ($str . ' temu');
	}
}
